/***************************************************************************
* 
* File              : valentine.c
*
* Author			: Kurt E. Clothier
* Date				: February 13, 2013
* Modified			: February 13, 2013
*
* Description       : Control LEDS in valentine box
*
* Compiler			: AVR-GCC
*
* Hardware			: ATTINY25; Calibrated Internal Oscillator with CKDIV8
*
* Fuse Settings		: DEFAULT (lfuse:w0x62:m hfuse:w0xdf:m efuse:w0xff:m)
*
****************************************************************************/
/**	ATTINY25 Summary *******************************************************
*
* --- Avialable Pins ---
*
*	Pin  |  # |    Function								|	Usage |	Direction
*  --------------------------------------------------------------	
*	PB5		1	/RESET/PCINT5/ADC0/dW						/RESET	NA
*	PB3		2	/OC1B/PCINT3/XTAL1/CLKI/ADC3				INTB	Input	
*	PB4		3	OC1B/PCINT4/XTAL2/CLKO/ADC2					SS		Output
*	GND		4	Ground										GND		NA
*	PB0		5	/OC1A/MOSI/DI/SDA/ANI0/OC0A/AREF/PCINT0		DI		Input
*	PB1		6	OC1A/MISO/DO/AIN1/OC0B/PCINT1				DO		Output
*	PB2		7	SCK/USCK/SCL/ADC1/T0/INT0/PCINT2			USCK	Output
*	VCC		8	Power										+5V		NA
*
****************************************************************************/
 
//-------------------------
//		Includes
//-------------------------
#include <custom/avr.h>

//-------------------------
//		Definitions
//-------------------------

#define LED0 _BV(PB0)
#define LED1 _BV(PB1)
#define LED2 _BV(PB2)
#define LED3 _BV(PB3)
#define LED4 _BV(PB4)

// Values for stat_flag
#define DIRECTION0	0x01
#define DIRECTION1	0x02
#define DIRECTION2	0x04
#define DIRECTION3	0x08
#define DIRECTION4	0x10

#define DELAY_MS	0x80

//-------------------------
//		  Macros
//-------------------------

#define LED_ON0		(PORTB |= LED0)
#define LED_ON1		(PORTB |= LED1)
#define LED_ON2		(PORTB |= LED2)
#define LED_ON3		(PORTB |= LED3)
#define LED_ON4		(PORTB |= LED4)

#define LED_OFF0	(PORTB &= ~LED0)
#define LED_OFF1	(PORTB &= ~LED1)
#define LED_OFF2	(PORTB &= ~LED2)
#define LED_OFF3	(PORTB &= ~LED3)
#define LED_OFF4	(PORTB &= ~LED4)

#define LED_ON(LED)		(LED_ON ## LED)
#define LED_OFF(LED)	(LED_OFF ## LED)

//-------------------------
//      Global Variables
//-------------------------

static volatile uint8_t 	stat_flag = 0x00;	// Status Flags

static volatile uint8_t		ns_delay_cnt = 0;	// 100ns delay counter
static volatile uint16_t	ms_delay_cnt = 0;	// 1ms delay counter
static volatile uint8_t		cnt0 = 0;			// Counter 0
static volatile uint8_t		cnt1 = 20;			// Counter 1
static volatile uint8_t		cnt2 = 40;			// Counter 2
static volatile uint8_t		cnt3 = 60;			// Counter 3
static volatile uint8_t		cnt4 = 80;			// Counter 4


static volatile uint8_t		brightness0 = 1;	// brightness level for LED0
static volatile uint8_t		brightness1 = 100;	// brightness level for LED1
static volatile uint8_t		brightness2 = 12;	// brightness level for LED2
static volatile uint8_t		brightness3 = 40;	// brightness level for LED3
static volatile uint8_t		brightness4 = 1;	// brightness level for LED4

//-------------------------
//      Local Functions
//-------------------------

void delay_ms(uint16_t ms);

//-------------------------
//		Main
//-------------------------

int main (void) 
{

  /****************************
	Initialize ATtiny25 Device
  ****************************/
  
  cli();	// Turn off interrupts
  
  // Power Reduction Register
  PRR = 
	_BV(PRTIM0) |	// Timer0 - 118.2uA
	_BV(PRTIM1) |	// Timer1 - 153.0uA
	_BV(PRUSI)  |	// USI	  - 92.2 uA
	_BV(PRADC);		// ADC	  - 333.3uA
	
  // Set I/O Ports
  DDRB = 0x1F;		// Set all as Outputs
  PORTB = 0x00;		// Set Outputs Low
  
  // Set up Timer 0 for 0.1ms interrupt
  PRR &= ~_BV(PRTIM0);			// Disable Timer Power Save
  TCCR0A = _BV(WGM01);			// CTC mode, TOP = OCRA, Outputs Disabled
  TCCR0B = _BV(CS00);			// Prescaler = 1
  OCR0A  = 99;					// Compare Value = (1MHz * 1ms / 1)-1
  TIMSK = _BV(OCIE0A);			// Enable Compare Match A Interrupt

  sei();						// Enable Interrupts
	
  for(;;)		// Enter "Forever Loop"
  {
    if(stat_flag & DIRECTION0){						// If LED 0 should increase in brightness
	  
	  if(brightness0 < 25) brightness0++;			// Increase slowly during dim states
	  else if(brightness0 < 40) brightness0 += 5;	// Increase slightly faster
	  else if(brightness0 < 100) brightness0 +=10;	// Increase rapidly
	  else CLEAR_BIT(stat_flag, DIRECTION0);		// Time to change direction
	}
	else{											// If LED 0 should decrease in brightness
	  if(brightness0 > 40) brightness0 -= 10;		// Decrease rapidly
	  else if(brightness0 > 25) brightness0 -= 5;	// Decrease slightly slower
	  else if(brightness0 > 1) brightness0--;		// Decrease slowly during dim states
	  else SET_BIT(stat_flag, DIRECTION0);			// Time to change direction
	}
	
	if(stat_flag & DIRECTION1){						// If LED 1 should increase in brightness
	  
	  if(brightness1 < 25) brightness1++;			// Increase slowly during dim states
	  else if(brightness1 < 40) brightness1 += 5;	// Increase slightly faster
	  else if(brightness1 < 100) brightness1 +=10;	// Increase rapidly
	  else CLEAR_BIT(stat_flag, DIRECTION1);		// Time to change direction
	}
	else{											// If LED 1 should decrease in brightness	
	  if(brightness1 > 40) brightness1 -= 10;		// Decrease rapidly
	  else if(brightness1 > 25) brightness1 -= 5;	// Decrease slightly slower
	  else if(brightness1 > 1) brightness1--;		// Decrease slowly during dim states
	  else SET_BIT(stat_flag, DIRECTION1);			// Time to change direction
	}
	
	if(stat_flag & DIRECTION2){										// If LED 2 should increase in brightness
	  if(++brightness2 >= 50) CLEAR_BIT(stat_flag, DIRECTION2);	// Increase slowly to 50%
	}
	else{															// If LED 2 should decrease in brightness
	  if(--brightness2 <= 1) SET_BIT(stat_flag, DIRECTION2);		// Decrease slowly to 1%
	}
	
	if(stat_flag & DIRECTION3){					// If LED 3 should increase in brightness
	  if(brightness3 < 50) brightness3 += 2;	// Increase quickly to 50%
	  else CLEAR_BIT(stat_flag, DIRECTION3);	// Change Direction
	}
	else{										// If LED 3 should decrease in brightness
	  if(brightness3 > 2) brightness3 -=2;		// Decrease quickly to 2%
	  else SET_BIT(stat_flag, DIRECTION3);		// Change Direction
	}
	
	if(stat_flag & DIRECTION4){										// If LED 4 should increase in brightness
	  if(++brightness4 >= 50) CLEAR_BIT(stat_flag, DIRECTION4);	// Increase slowly to 50%
	}
	else{															// If LED 2 should decrease in brightness
	  if(--brightness4 <= 1) SET_BIT(stat_flag, DIRECTION4);		// Decrease slowly to 1%
	}

	
	delay_ms(200);	// Delay for 200ms - this is how long LEDs are at each brightness level
	
  }
}

/**************************************************************************
	INTERRUPT HANDLERS
***************************************************************************/
// Catch-All Default for Unexpected Interrupts
//ISR(BADISR_vect){}

/***************************************************************
 * Timer/Counter0 Compare Match A - 1 ms timer
 *
 *	This ISR is the core timing mechanism of the entire program.
 *	It triggers every 1ms, and various actions are taken within.
 ***************************************************************/
ISR(TIMER0_COMPA_vect)
{
 /**************************************
  *     millisecond delay timer		   *
  **************************************/
  if(ms_delay_cnt){						// if need to count ms
	if(++ns_delay_cnt == 10){			// 10 * 100ns = 1ms
	  ms_delay_cnt--;					// decrement ms timer
	  ns_delay_cnt = 0;					// reset 100ns counter
	}
  }
  else CLEAR_BIT(stat_flag, DELAY_MS);	// when done, reset ms flag
  
 /**************************************
  *     LED Brightness Control		   *
  **************************************/
  if(++cnt0 == 100){	// 100 * 100ns = 10ms (100Hz)
	LED_ON(0);			// Turn on LED 0 
	cnt0 = 0;			// Reset Counter
  }
  else if (cnt0 == brightness0) LED_OFF(0);	// If reached desired brightness level, Turn off LED 0
  
  if(++cnt1 == 100){	// 100 * 100ns = 10ms (100Hz)
	LED_ON(1);
	cnt1 = 0;
  }
  else if (cnt1 == brightness1) LED_OFF(1);
  
  if(++cnt2 == 100){	// 100 * 100ns = 10ms (100Hz)
	LED_ON(2);
	cnt2 = 0;
  }
  else if (cnt2 == brightness2) LED_OFF(2);
  
  if(++cnt3 == 100){	// 100 * 100ns = 10ms (100Hz)
	LED_ON(3);
	cnt3 = 0;
  }
  else if (cnt3 == brightness3) LED_OFF(3);
  
  if(++cnt4 == 100){	// 100 * 100ns = 10ms (100Hz)
	LED_ON(4);
	cnt4 = 0;
  }
  else if (cnt4 == brightness4) LED_OFF(4);

}

/**************************************************************************
	DELAY ROUTINES AND UTILITIES
***************************************************************************/
// Delay for <ms> number of milliseconds
void delay_ms(uint16_t ms)
{
  SET_BIT(stat_flag, DELAY_MS);					// Set to enter loop
  ms_delay_cnt = ms;							// Triggers ISR
  LOOP_UNTIL_BIT_LO(stat_flag, DELAY_MS);		// Loop indefinitely
}

